<?php
/**
 * Plugin Name: Botamation WooCommerce Tools
 * Plugin URI: https://botamation.in
 * Description: Comprehensive WooCommerce webhook toolkit for abandoned cart recovery and order lifecycle notifications with configurable delays and actions.
 * Version: 1.0.1
 * Author: Botamation
 * Author URI: https://botamation.in
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: botamation-woocommerce-tools
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * WC requires at least: 3.0
 * WC tested up to: 8.2
 * Network: false
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('BWC_PLUGIN_VERSION', '1.0.1');
define('BWC_PLUGIN_FILE', __FILE__);
define('BWC_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('BWC_PLUGIN_URL', plugin_dir_url(__FILE__));
define('BWC_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main plugin class
 */
final class Botamation_WooCommerce_Tools {
    
    /**
     * Plugin instance
     *
     * @var Botamation_WooCommerce_Tools
     */
    private static $instance = null;

    /**
     * Get plugin instance
     *
     * @return Botamation_WooCommerce_Tools
     */
    public static function instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('plugins_loaded', array($this, 'init'), 10);
        // Declare compatibility with WooCommerce features (HPOS & Blocks)
        add_action('before_woocommerce_init', array($this, 'declare_wc_compatibility'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        register_uninstall_hook(__FILE__, array(__CLASS__, 'uninstall'));
    }

    /**
     * Initialize the plugin
     */
    public function init() {
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            return;
        }

        // Load plugin files
        $this->load_plugin_textdomain();
        $this->includes();
        $this->init_hooks();
    }

    /**
     * Declare compatibility with WooCommerce features to avoid admin warnings.
     * - custom_order_tables (HPOS)
     * - cart_checkout_blocks (Checkout block-based experience)
     */
    public function declare_wc_compatibility() {
        if (class_exists('\\Automattic\\WooCommerce\\Utilities\\FeaturesUtil')) {
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', BWC_PLUGIN_FILE, true);
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('cart_checkout_blocks', BWC_PLUGIN_FILE, true);
        }
    }

    /**
     * Load plugin text domain for translations
     */
    public function load_plugin_textdomain() {
        load_plugin_textdomain(
            'botamation-woocommerce-tools',
            false,
            dirname(BWC_PLUGIN_BASENAME) . '/languages/'
        );
    }

    /**
     * Include required files
     */
    private function includes() {
        require_once BWC_PLUGIN_DIR . 'includes/class-abandoned-cart.php';
        require_once BWC_PLUGIN_DIR . 'includes/class-order-notification.php';
        require_once BWC_PLUGIN_DIR . 'includes/class-admin.php';
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Initialize modules
        BWC_Abandoned_Cart::instance();
        BWC_Order_Notification::instance();
        BWC_Admin::instance();
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Create database tables
        $this->create_tables();
        
        // Set default options
        $this->set_default_options();
        
        // Clear any cached data
        wp_cache_flush();
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clear scheduled crons
        wp_clear_scheduled_hook('bwc_process_abandoned_carts');
        wp_clear_scheduled_hook('bwc_cleanup_old_records');
        
        // Clear any cached data
        wp_cache_flush();
    }

    /**
     * Plugin uninstall
     */
    public static function uninstall() {
        // Remove plugin options
        delete_option('bwc_abandoned_cart_settings');
        delete_option('bwc_order_notification_settings');
        
        // Drop database tables
        global $wpdb;
        $table_name = $wpdb->prefix . 'bwc_abandoned_carts';
        $wpdb->query("DROP TABLE IF EXISTS {$table_name}");
        
        // Clear any cached data
        wp_cache_flush();
    }

    /**
     * Create database tables
     */
    private function create_tables() {
        // Delegate to the Abandoned Cart module to ensure schema parity
        if (!class_exists('BWC_Abandoned_Cart')) {
            require_once BWC_PLUGIN_DIR . 'includes/class-abandoned-cart.php';
        }
        if (class_exists('BWC_Abandoned_Cart')) {
            BWC_Abandoned_Cart::create_table();
        }
    }

    /**
     * Set default options
     */
    private function set_default_options() {
        // Abandoned cart default settings
        $default_settings = array(
            'webhook_url' => '',
            'remind_minutes' => 15
        );
        
        if (!get_option('bwc_abandoned_cart_settings')) {
            add_option('bwc_abandoned_cart_settings', $default_settings);
        }
        
        // Order notification default settings
        if (!get_option('bwc_order_notification_settings')) {
            add_option('bwc_order_notification_settings', array());
        }
    }

    /**
     * WooCommerce missing notice
     */
    public function woocommerce_missing_notice() {
        $class = 'notice notice-error';
        $message = sprintf(
            /* translators: %s: Plugin name */
            __('The %s plugin requires WooCommerce to be installed and active.', 'botamation-woocommerce-tools'),
            '<strong>Botamation WooCommerce Tools</strong>'
        );
        
        printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
    }
}

/**
 * Returns the main instance of Botamation_WooCommerce_Tools
 *
 * @return Botamation_WooCommerce_Tools
 */
function BWC() {
    return Botamation_WooCommerce_Tools::instance();
}

// Initialize the plugin
BWC();