<?php
/**
 * Abandoned Cart functionality
 *
 * @package Botamation_WooCommerce_Tools
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * BWC_Abandoned_Cart class
 */
class BWC_Abandoned_Cart {

    /**
     * Instance
     *
     * @var BWC_Abandoned_Cart
     */
    private static $instance = null;

    /**
     * Constants
     */
    const OPTION_NAME = 'bwc_abandoned_cart_settings';
    const OPTION_GROUP = 'bwc_abandoned_cart_group';
    const CRON_HOOK = 'bwc_process_abandoned_carts';
    const CLEANUP_CRON_HOOK = 'bwc_cleanup_old_records';
    const TABLE_SLUG = 'bwc_abandoned_carts';
    const DEFAULT_REMIND = 15;
    const MIN_REMIND = 1;
    const NONCE_ACTION = 'bwc_guest_details';
    const RESTORE_QUERY_ARG = 'bwc_restore';
    const SESSION_PHONE_KEY = 'bwc_guest_phone';
    const SESSION_RESTORE_KEY = 'bwc_restore_token';
    const SESSION_BILLING_CACHE = 'bwc_billing_cache';
    const SESSION_SHIPPING_CACHE = 'bwc_shipping_cache';

    /**
     * Get instance
     *
     * @return BWC_Abandoned_Cart
     */
    public static function instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('wp_ajax_bwc_update_guest_details', array($this, 'handle_guest_details'));
        add_action('wp_ajax_nopriv_bwc_update_guest_details', array($this, 'handle_guest_details'));
        add_action('template_redirect', array($this, 'maybe_restore_cart'));
        add_action('woocommerce_cart_updated', array($this, 'capture_cart'), 20);
        add_action('woocommerce_add_to_cart', array($this, 'capture_cart'), 20);
        add_action('woocommerce_cart_emptied', array($this, 'clear_cart'), 20);
        add_action('woocommerce_checkout_order_processed', array($this, 'mark_cart_converted'), 20, 1);
        add_action(self::CRON_HOOK, array($this, 'handle_cron'));
        add_action(self::CLEANUP_CRON_HOOK, array($this, 'handle_cleanup_cron'));
        
        // Schedule events
        $this->schedule_events();
    }

    /**
     * Activation hook
     */
    public function activate() {
        self::create_table();
        $this->schedule_events();
    }

    /**
     * Deactivation hook
     */
    public function deactivate() {
        wp_clear_scheduled_hook(self::CRON_HOOK);
        wp_clear_scheduled_hook(self::CLEANUP_CRON_HOOK);
    }

    /**
     * Create DB table
     */
    public static function create_table() {
        global $wpdb;
        $table_name = $wpdb->prefix . self::TABLE_SLUG;
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table_name} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            session_key varchar(191) NOT NULL,
            user_id bigint(20) unsigned NOT NULL DEFAULT 0,
            cart_hash char(32) NOT NULL,
            cart_items longtext NOT NULL,
            billing longtext,
            shipping longtext,
            contact_phone varchar(64) DEFAULT NULL,
            contact_email varchar(191) DEFAULT NULL,
            restore_key char(32) NOT NULL,
            status varchar(20) NOT NULL DEFAULT 'pending',
            last_updated datetime NOT NULL,
            sent_at datetime DEFAULT NULL,
            last_attempt datetime DEFAULT NULL,
            last_error text,
            PRIMARY KEY  (id),
            UNIQUE KEY session_key (session_key),
            UNIQUE KEY restore_key (restore_key),
            KEY status_last_updated (status,last_updated)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Schedule cron events
     */
    public function schedule_events() {
        if (!wp_next_scheduled(self::CRON_HOOK)) {
            $this->schedule_event();
        }
        
        if (!wp_next_scheduled(self::CLEANUP_CRON_HOOK)) {
            $this->schedule_cleanup_event();
        }
    }

    /**
     * Schedule the main abandoned cart cron as a single event based on settings
     */
    private function schedule_event() {
        $settings = $this->get_settings();
        $minutes = isset($settings['remind_minutes']) ? absint($settings['remind_minutes']) : self::DEFAULT_REMIND;
        $minutes = max(self::MIN_REMIND, $minutes);

        if (!wp_next_scheduled(self::CRON_HOOK)) {
            wp_schedule_single_event(time() + ($minutes * MINUTE_IN_SECONDS), self::CRON_HOOK);
        }
    }

    /**
     * Schedule daily cleanup cron
     */
    private function schedule_cleanup_event() {
        if (!wp_next_scheduled(self::CLEANUP_CRON_HOOK)) {
            wp_schedule_event(time() + HOUR_IN_SECONDS, 'daily', self::CLEANUP_CRON_HOOK);
        }
    }

    /**
     * Enqueue assets
     */
    public function enqueue_assets() {
        if (!is_admin() && function_exists('is_checkout') && is_checkout()) {
            wp_enqueue_script(
                'bwc-checkout',
                BWC_PLUGIN_URL . 'assets/js/wcf-checkout.js',
                array('jquery'),
                BWC_PLUGIN_VERSION,
                true
            );

            wp_localize_script('bwc-checkout', 'bwc_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce(self::NONCE_ACTION),
            ));
        }
    }

    /**
     * Handle guest details AJAX
     */
    public function handle_guest_details() {
        if (!wp_verify_nonce($_POST['nonce'] ?? '', self::NONCE_ACTION)) {
            wp_die(__('Security check failed.', 'botamation-woocommerce-tools'));
        }

        // Sanitize inputs from both billing and shipping
        $billing_phone = sanitize_text_field($_POST['billing_phone'] ?? '');
        $shipping_phone = sanitize_text_field($_POST['shipping_phone'] ?? '');
        $billing_first = sanitize_text_field($_POST['billing_first_name'] ?? '');
        $billing_last  = sanitize_text_field($_POST['billing_last_name'] ?? '');
        $billing_email = sanitize_email($_POST['billing_email'] ?? '');
        $shipping_first = sanitize_text_field($_POST['shipping_first_name'] ?? '');
        $shipping_last  = sanitize_text_field($_POST['shipping_last_name'] ?? '');
        $shipping_email = sanitize_email($_POST['shipping_email'] ?? '');

        // Prefer billing phone, fallback to shipping phone
        $phone = $billing_phone ?: $shipping_phone;
        if (!empty($phone)) {
            WC()->session->set(self::SESSION_PHONE_KEY, $phone);
        }

        // Cache partial billing details for guests so capture_cart can use them
        if ($billing_first || $billing_last || $billing_email || $billing_phone) {
            WC()->session->set(self::SESSION_BILLING_CACHE, array(
                'first_name' => $billing_first,
                'last_name'  => $billing_last,
                'email'      => $billing_email,
                'phone'      => $billing_phone,
            ));
        }

        // Cache partial shipping details as well
        if ($shipping_first || $shipping_last || $shipping_email || $shipping_phone) {
            WC()->session->set(self::SESSION_SHIPPING_CACHE, array(
                'first_name' => $shipping_first,
                'last_name'  => $shipping_last,
                'email'      => $shipping_email,
                'phone'      => $shipping_phone,
            ));
        }

        // Attempt to capture cart immediately
        $this->capture_cart();

        wp_send_json_success(array(
            'message' => __('Guest details saved.', 'botamation-woocommerce-tools')
        ));
    }

    /**
     * Maybe restore cart from URL parameter
     */
    public function maybe_restore_cart() {
        if (!isset($_GET[self::RESTORE_QUERY_ARG])) {
            return;
        }

        if (!function_exists('WC') || !WC()->cart || !WC()->session) {
            return;
        }

        $restore_key = sanitize_text_field(wp_unslash($_GET[self::RESTORE_QUERY_ARG]));
        if (!$restore_key) {
            return;
        }

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SLUG;
        $row = $wpdb->get_row($wpdb->prepare("SELECT cart_items FROM {$table} WHERE restore_key = %s", $restore_key), ARRAY_A);

        if (!$row) {
            wc_add_notice(__('This checkout link has expired.', 'botamation-woocommerce-tools'), 'error');
            return;
        }

        $items = json_decode($row['cart_items'], true);
        if (!is_array($items)) {
            return;
        }

        WC()->cart->empty_cart();
        foreach ($items as $item) {
            $product_id = isset($item['product_id']) ? absint($item['product_id']) : 0;
            if (!$product_id) {
                continue;
            }
            $quantity = isset($item['quantity']) ? max(1, absint($item['quantity'])) : 1;
            $variation_id = isset($item['variation_id']) ? absint($item['variation_id']) : 0;
            $variation = isset($item['variation']) && is_array($item['variation']) ? array_map('wc_clean', $item['variation']) : array();
            WC()->cart->add_to_cart($product_id, $quantity, $variation_id, $variation);
        }

        WC()->session->set(self::SESSION_RESTORE_KEY, $restore_key);
        wc_add_notice(__('Your cart has been restored. You can complete checkout below.', 'botamation-woocommerce-tools'), 'success');

        wp_safe_redirect(remove_query_arg(self::RESTORE_QUERY_ARG));
        exit;
    }

    /**
     * Capture cart data
     */
    public function capture_cart() {
        if (is_admin() || !function_exists('WC') || !WC()->cart) {
            return;
        }

        $cart = WC()->cart;
        if ($cart->is_empty()) {
            return;
        }

        $user_id = get_current_user_id();
        $session_id = WC()->session->get_customer_id();
        $cart_id = $user_id ? "user_{$user_id}" : "guest_{$session_id}";

        if (empty($cart_id)) {
            return;
        }

        $this->process_session_cart($cart_id, $session_id, $user_id);
    }

    /**
     * Process session cart
     */
    private function process_session_cart($cart_id, $session_id, $user_id) {
        $cart = WC()->cart;
        $items = array();

        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            $product = $cart_item['data'];
            if (!$product || !$product->exists()) {
                continue;
            }

            $image_url = '';
            if ($product->get_image_id()) {
                $image_url = wp_get_attachment_image_url($product->get_image_id(), 'full');
            }
            if (!$image_url) {
                $image_url = wc_placeholder_img_src();
            }

            $items[] = array(
                'product_id' => $product->get_id(),
                'variation_id' => $cart_item['variation_id'] ?? 0,
                'quantity' => $cart_item['quantity'],
                'name' => $product->get_name(),
                'price' => floatval($product->get_price()),
                'subtotal' => floatval($cart_item['line_subtotal']),
                'image' => $image_url,
                'variation' => $cart_item['variation'] ?? array(),
            );
        }

        if (empty($items)) {
            return;
        }

        // Get contact data
        $contact_data = $this->get_contact_data($user_id);
        
        // Don't save record if phone is not available
        if (empty($contact_data['phone'])) {
            return;
        }

        $cart_data_items = array(
            'items' => $items,
            'cart_total' => floatval($cart->get_total('edit')),
            'cart_subtotal' => floatval($cart->get_subtotal()),
            'cart_tax' => floatval($cart->get_total_tax()),
        );

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SLUG;
        $now = current_time('mysql', true);
        $hash = md5(wp_json_encode($items));

        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT id, cart_hash, restore_key, status FROM {$table} WHERE session_key = %s",
            $session_id
        ));

        $restore_key = WC()->session->get(self::SESSION_RESTORE_KEY);
        if (!$restore_key && $existing) {
            $restore_key = $existing->restore_key;
        }
        if (!$restore_key) {
            $restore_key = bin2hex(random_bytes(16));
            WC()->session->set(self::SESSION_RESTORE_KEY, $restore_key);
        }

        $data = [
            'user_id' => $user_id,
            'cart_hash' => $hash,
            'cart_items' => wp_json_encode($items),
            'billing' => wp_json_encode($this->get_billing_data($user_id)),
            'shipping' => wp_json_encode($this->get_shipping_data($user_id)),
            'contact_phone' => $contact_data['phone'],
            'contact_email' => $contact_data['email'],
            'restore_key' => $restore_key,
        ];

        if ($existing) {
            if ($existing->cart_hash !== $hash || in_array($existing->status, ['sent', 'error', 'expired', 'converted'], true)) {
                $data['status'] = 'pending';
                $data['sent_at'] = null;
                $data['last_attempt'] = null;
                $data['last_error'] = null;
                $data['last_updated'] = $now;
                $wpdb->update($table, $data, ['id' => $existing->id]);
            } else {
                // Only update contact info, don't change timestamp
                $wpdb->update($table, $data, ['id' => $existing->id]);
            }
        } else {
            $data['session_key'] = $session_id;
            $data['status'] = 'pending';
            $data['last_updated'] = $now;
            $wpdb->insert($table, $data);
        }
    }

    /**
     * Get contact data
     */
    private function get_contact_data($user_id) {
        $contact = array('phone' => '', 'email' => '');

        if ($user_id) {
            $user = get_userdata($user_id);
            if ($user) {
                $contact['email'] = $user->user_email;
                $contact['phone'] = get_user_meta($user_id, 'billing_phone', true);
            }
        } else {
            // Guest: prioritize our cached session data over WC customer object
            // because WC customer object may not be populated until checkout submit
            $contact['phone'] = WC()->session->get(self::SESSION_PHONE_KEY, '');
            
            // Try billing cache first
            $billing_cache = WC()->session->get(self::SESSION_BILLING_CACHE, array());
            if (is_array($billing_cache) && !empty($billing_cache['email'])) {
                $contact['email'] = $billing_cache['email'];
            }
            
            // Fallback to shipping cache
            if (empty($contact['email'])) {
                $shipping_cache = WC()->session->get(self::SESSION_SHIPPING_CACHE, array());
                if (is_array($shipping_cache) && !empty($shipping_cache['email'])) {
                    $contact['email'] = $shipping_cache['email'];
                }
            }
            
            // Last resort: try WooCommerce customer object
            if (empty($contact['email']) && WC()->customer) {
                $wc_email = WC()->customer->get_billing_email();
                if (!empty($wc_email)) {
                    $contact['email'] = $wc_email;
                }
            }
        }

        return $contact;
    }

    /**
     * Get billing data
     */
    private function get_billing_data($user_id) {
        $billing = array();
        
        if ($user_id) {
            $fields = array('first_name', 'last_name', 'company', 'email', 'phone', 
                          'address_1', 'address_2', 'city', 'state', 'postcode', 'country');
            
            foreach ($fields as $field) {
                $billing[$field] = get_user_meta($user_id, 'billing_' . $field, true);
            }
        } else {
            // Guest: merge WooCommerce customer data with cached session inputs
            if (WC()->customer) {
                $billing = array(
                    'first_name' => WC()->customer->get_billing_first_name(),
                    'last_name'  => WC()->customer->get_billing_last_name(),
                    'company'    => WC()->customer->get_billing_company(),
                    'email'      => WC()->customer->get_billing_email(),
                    'phone'      => WC()->customer->get_billing_phone(),
                    'address_1'  => WC()->customer->get_billing_address_1(),
                    'address_2'  => WC()->customer->get_billing_address_2(),
                    'city'       => WC()->customer->get_billing_city(),
                    'state'      => WC()->customer->get_billing_state(),
                    'postcode'   => WC()->customer->get_billing_postcode(),
                    'country'    => WC()->customer->get_billing_country(),
                );
            }
            
            // Overlay with cached session data (which may have earlier inputs)
            $cache = WC()->session->get(self::SESSION_BILLING_CACHE, array());
            if (is_array($cache)) {
                foreach ($cache as $key => $value) {
                    if (!empty($value)) {
                        $billing[$key] = $value;
                    }
                }
            }
        }

        return $billing;
    }

    /**
     * Get shipping data
     */
    private function get_shipping_data($user_id) {
        $shipping = array();
        
        if ($user_id) {
            $fields = array('first_name', 'last_name', 'company', 'address_1', 
                          'address_2', 'city', 'state', 'postcode', 'country');
            
            foreach ($fields as $field) {
                $shipping[$field] = get_user_meta($user_id, 'shipping_' . $field, true);
            }
        } else {
            // Guest: merge WooCommerce customer data with cached session inputs
            if (WC()->customer) {
                $shipping = array(
                    'first_name' => WC()->customer->get_shipping_first_name(),
                    'last_name'  => WC()->customer->get_shipping_last_name(),
                    'company'    => WC()->customer->get_shipping_company(),
                    'address_1'  => WC()->customer->get_shipping_address_1(),
                    'address_2'  => WC()->customer->get_shipping_address_2(),
                    'city'       => WC()->customer->get_shipping_city(),
                    'state'      => WC()->customer->get_shipping_state(),
                    'postcode'   => WC()->customer->get_shipping_postcode(),
                    'country'    => WC()->customer->get_shipping_country(),
                );
            }
            
            // Overlay with cached session data
            $cache = WC()->session->get(self::SESSION_SHIPPING_CACHE, array());
            if (is_array($cache)) {
                foreach ($cache as $key => $value) {
                    if (!empty($value)) {
                        $shipping[$key] = $value;
                    }
                }
            }
        }

        return $shipping;
    }

    /**
     * Clear cart
     */
    public function clear_cart() {
        $user_id = get_current_user_id();
        $session_id = WC()->session->get_customer_id();
        $cart_id = $user_id ? "user_{$user_id}" : "guest_{$session_id}";

        if (!$cart_id) {
            return;
        }

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SLUG;

        $wpdb->delete($table, ['session_key' => $session_id]);
    }

    /**
     * Mark cart as converted
     */
    public function mark_cart_converted($order_id) {
        $user_id = get_current_user_id();
        $session_id = WC()->session->get_customer_id();
        $cart_id = $user_id ? "user_{$user_id}" : "guest_{$session_id}";

        if (!$cart_id) {
            return;
        }

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SLUG;

        $wpdb->update(
            $table,
            array('status' => 'converted', 'last_updated' => current_time('mysql', true)),
            array('session_key' => $session_id),
            array('%s', '%s'),
            array('%s')
        );
    }

    /**
     * Handle cron job
     */
    public function handle_cron() {
        $settings = $this->get_settings();
        $webhook_url = $settings['webhook_url'] ?? '';
        $remind_minutes = $settings['remind_minutes'] ?? self::DEFAULT_REMIND;

        if (empty($webhook_url)) {
            error_log('BWC Abandoned Cart: No webhook URL configured');
            return;
        }

        $cutoff = gmdate('Y-m-d H:i:s', current_time('timestamp', true) - ($remind_minutes * 60));

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SLUG;

        // Select eligible carts first
        $eligible_carts = $wpdb->get_results($wpdb->prepare(
            "SELECT * 
             FROM {$table} 
             WHERE status = 'pending' AND last_updated < %s 
             ORDER BY last_updated ASC 
             LIMIT 50",
            $cutoff
        ));

        if (empty($eligible_carts)) {
            error_log('BWC Abandoned Cart: No eligible carts found');
            $this->schedule_event();
            return;
        }

        error_log('BWC Abandoned Cart: Processing ' . count($eligible_carts) . ' eligible carts');

        // Capture currently active carts to avoid processing them
        $this->capture_cart();

        // Process eligible carts
        foreach ($eligible_carts as $cart_record) {
            $this->process_cart_record($cart_record, $webhook_url);
        }

        // Schedule next event
        $this->schedule_event();
    }

    /**
     * Process individual cart record
     */
    private function process_cart_record($cart_record, $webhook_url) {
        // Build payload directly from row (JSON-decoded in build_payload)
        $payload = $this->build_payload((array) $cart_record);

        $response = $this->send_webhook($webhook_url, $payload);

        // Update cart status and attempt metadata
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SLUG;

        $data = array(
            'last_attempt' => current_time('mysql', true),
        );

        if (is_wp_error($response)) {
            $data['status'] = 'error';
            $data['last_error'] = $response->get_error_message();
        } else {
            $code = (int) wp_remote_retrieve_response_code($response);
            if ($code >= 200 && $code < 300) {
                $data['status'] = 'sent';
                $data['sent_at'] = current_time('mysql', true);
                $data['last_error'] = '';
            } else {
                $data['status'] = 'error';
                $data['last_error'] = wp_remote_retrieve_body($response);
            }
        }

        $wpdb->update(
            $table,
            $data,
            array('id' => $cart_record->id),
            null,
            array('%d')
        );
    }

    /**
     * Send webhook
     */
    private function send_webhook($url, $payload) {
        $response = wp_remote_post($url, array(
            'method' => 'POST',
            'timeout' => 45,
            'redirection' => 5,
            'httpversion' => '1.0',
            'blocking' => true,
            'headers' => array('Content-Type' => 'application/json'),
            'body' => wp_json_encode($payload),
            'cookies' => array()
        ));

        if (is_wp_error($response)) {
            error_log('BWC Webhook Error: ' . $response->get_error_message());
        } else {
            $code = (int) wp_remote_retrieve_response_code($response);
            error_log('BWC Webhook Response Code: ' . $code);
        }

        return $response;
    }

    /**
     * Generate restore URL
     */
    private function generate_restore_url($cart_record) {
        // Not used by current flow; keep for backward compatibility if needed
        return add_query_arg(self::RESTORE_QUERY_ARG, $cart_record->restore_key ?? '', wc_get_checkout_url());
    }

    /**
     * Handle cleanup cron
     */
    public function handle_cleanup_cron() {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SLUG;
        $cutoff = gmdate('Y-m-d H:i:s', current_time('timestamp', true) - (60 * HOUR_IN_SECONDS));
        $wpdb->query($wpdb->prepare("DELETE FROM {$table} WHERE last_updated < %s", $cutoff));
    }

    /**
     * Get settings
     */
    public function get_settings() {
        $defaults = array(
            'webhook_url' => '',
            'remind_minutes' => self::DEFAULT_REMIND,
        );
        return wp_parse_args(get_option(self::OPTION_NAME, array()), $defaults);
    }

    /**
     * Build payload for webhook
     */
    private function build_payload(array $row): array {
        $billing = isset($row['billing']) ? json_decode($row['billing'], true) : [];
        $shipping = isset($row['shipping']) ? json_decode($row['shipping'], true) : [];
        $items = isset($row['cart_items']) ? json_decode($row['cart_items'], true) : [];

        // Build contact object - prefer billing, fallback to shipping for each field
        $contact = [
            'phone' => '',
            'email' => '',
            'first_name' => '',
            'last_name' => '',
        ];

        // Phone: prefer contact_phone (from billing), then billing phone, then shipping phone
        $contact['phone'] = $row['contact_phone'] ?? '';
        if (empty($contact['phone']) && is_array($billing)) {
            $contact['phone'] = $billing['phone'] ?? '';
        }
        if (empty($contact['phone']) && is_array($shipping)) {
            $contact['phone'] = $shipping['phone'] ?? '';
        }

        // Email: prefer contact_email (from billing), then billing email, then shipping email
        $contact['email'] = $row['contact_email'] ?? '';
        if (empty($contact['email']) && is_array($billing)) {
            $contact['email'] = $billing['email'] ?? '';
        }
        if (empty($contact['email']) && is_array($shipping)) {
            $contact['email'] = $shipping['email'] ?? '';
        }

        // First name: prefer billing, fallback to shipping
        if (is_array($billing)) {
            $contact['first_name'] = $billing['first_name'] ?? '';
        }
        if (empty($contact['first_name']) && is_array($shipping)) {
            $contact['first_name'] = $shipping['first_name'] ?? '';
        }

        // Last name: prefer billing, fallback to shipping
        if (is_array($billing)) {
            $contact['last_name'] = $billing['last_name'] ?? '';
        }
        if (empty($contact['last_name']) && is_array($shipping)) {
            $contact['last_name'] = $shipping['last_name'] ?? '';
        }

        return [
            'generated_at' => current_time('mysql'),
            'session_key' => $row['session_key'],
            'restore_key' => $row['restore_key'],
            'checkout_url' => add_query_arg(self::RESTORE_QUERY_ARG, rawurlencode($row['restore_key']), wc_get_checkout_url()),
            'billing' => $billing ?: new stdClass(),
            'shipping' => $shipping ?: new stdClass(),
            'contact' => $contact,
            'items' => $items ?: [],
        ];
    }

    /**
     * Send test webhook
     */
    public function send_test_webhook() {
        $settings = $this->get_settings();
        
        if (empty($settings['webhook_url'])) {
            return array(
                'success' => false,
                'message' => __('Please configure a webhook URL first.', 'botamation-woocommerce-tools')
            );
        }

        // Build a synthetic test payload matching the exact format we use in build_payload
        $test_restore_key = wp_generate_password(32, false);
        $test_session_key = 'test_' . wp_generate_password(32, false);
        
        $test_payload = array(
            'test' => true,
            'generated_at' => current_time('mysql'),
            'session_key' => $test_session_key,
            'restore_key' => $test_restore_key,
            'checkout_url' => add_query_arg(self::RESTORE_QUERY_ARG, rawurlencode($test_restore_key), wc_get_checkout_url()),
            'billing' => array(
                'first_name' => 'John',
                'last_name' => 'Doe',
                'company' => 'Test Company',
                'email' => 'john.doe@example.com',
                'phone' => '+1234567890',
                'address_1' => '123 Test Street',
                'address_2' => 'Suite 100',
                'city' => 'Test City',
                'state' => 'TS',
                'postcode' => '12345',
                'country' => 'US',
            ),
            'shipping' => array(
                'first_name' => 'Jane',
                'last_name' => 'Smith',
                'company' => 'Shipping Co',
                'email' => 'jane.smith@example.com',
                'phone' => '+0987654321',
                'address_1' => '456 Ship Lane',
                'address_2' => '',
                'city' => 'Ship City',
                'state' => 'SC',
                'postcode' => '67890',
                'country' => 'US',
            ),
            'contact' => array(
                'phone' => '+1234567890',
                'email' => 'john.doe@example.com',
                'first_name' => 'John',
                'last_name' => 'Doe',
            ),
            'items' => array(
                array(
                    'product_id' => 1,
                    'variation_id' => 0,
                    'name' => 'Sample Product',
                    'price' => 29.99,
                    'quantity' => 2,
                    'subtotal' => 59.98,
                    'image' => 'https://via.placeholder.com/300x300.jpg',
                    'variation' => array(),
                ),
                array(
                    'product_id' => 2,
                    'variation_id' => 5,
                    'name' => 'Variant Product - Size: Large, Color: Blue',
                    'price' => 19.99,
                    'quantity' => 1,
                    'subtotal' => 19.99,
                    'image' => 'https://via.placeholder.com/300x300.jpg',
                    'variation' => array(
                        'Size' => 'Large',
                        'Color' => 'Blue',
                    ),
                ),
            ),
        );

        $body = wp_json_encode($test_payload);
        if (false === $body) {
            return array(
                'success' => false,
                'message' => __('Failed to encode test payload.', 'botamation-woocommerce-tools')
            );
        }

        $response = wp_remote_post($settings['webhook_url'], array(
            'timeout' => 15,
            'headers' => array(
                'Content-Type' => 'application/json',
                'User-Agent' => 'BWC-Abandoned-Cart-Test-Webhook/1.0'
            ),
            'body' => $body,
        ));

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => sprintf(__('Webhook test failed: %s', 'botamation-woocommerce-tools'), $response->get_error_message())
            );
        }

        $code = (int) wp_remote_retrieve_response_code($response);
        if ($code >= 200 && $code < 300) {
            return array(
                'success' => true,
                'message' => sprintf(__('Test webhook sent successfully! Response code: %d', 'botamation-woocommerce-tools'), $code)
            );
        } else {
            $body_response = wp_remote_retrieve_body($response);
            return array(
                'success' => false,
                'message' => sprintf(__('Webhook returned error code %d: %s', 'botamation-woocommerce-tools'), $code, $body_response)
            );
        }
    }
}