<?php
/**
 * Admin functionality
 *
 * @package Botamation_WooCommerce_Tools
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * BWC_Admin class
 */
class BWC_Admin {

    /**
     * Instance
     *
     * @var BWC_Admin
     */
    private static $instance = null;

    /**
     * Get instance
     *
     * @return BWC_Admin
     */
    public static function instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('admin_menu', array($this, 'register_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
    }

    /**
     * Register admin menu
     */
    public function register_admin_menu() {
        // Top-level menu: Botamation Tools
        add_menu_page(
            __('Botamation Tools', 'botamation-woocommerce-tools'),
            __('Botamation Tools', 'botamation-woocommerce-tools'),
            'manage_options',
            'botamation-tools',
            array($this, 'render_tools_page'),
            'dashicons-admin-tools',
            56
        );

        // Submenu: Abandoned Cart Notification
        add_submenu_page(
            'botamation-tools',
            __('Abandoned Cart Notification', 'botamation-woocommerce-tools'),
            __('Abandoned Cart Notification', 'botamation-woocommerce-tools'),
            'manage_options',
            'bwc-abandoned-cart',
            array($this, 'render_abandoned_cart_page')
        );

        // Submenu: Order Notification
        add_submenu_page(
            'botamation-tools',
            __('Order Notification', 'botamation-woocommerce-tools'),
            __('Order Notification', 'botamation-woocommerce-tools'),
            'manage_options',
            'bwc-order-notification',
            array($this, 'render_order_notification_page')
        );

        // Hide the duplicate parent submenu entry that WP adds automatically
        remove_submenu_page('botamation-tools', 'botamation-tools');
    }

    /**
     * Register settings
     */
    public function register_settings() {
        // Abandoned Cart settings
        register_setting(
            BWC_Abandoned_Cart::OPTION_GROUP,
            BWC_Abandoned_Cart::OPTION_NAME,
            array($this, 'sanitize_abandoned_cart_settings')
        );

        // Order Notification settings
        register_setting(
            BWC_Order_Notification::OPTION_GROUP,
            BWC_Order_Notification::OPTION_NAME,
            array($this, 'sanitize_order_notification_settings')
        );
    }

    /**
     * Sanitize abandoned cart settings
     */
    public function sanitize_abandoned_cart_settings($input) {
        if (!is_array($input)) {
            $input = array();
        }

        $settings = BWC_Abandoned_Cart::instance()->get_settings();

        if (isset($input['webhook_url'])) {
            $settings['webhook_url'] = esc_url_raw(trim($input['webhook_url']));
        }

        $minutes = isset($input['remind_minutes']) ? absint($input['remind_minutes']) : BWC_Abandoned_Cart::DEFAULT_REMIND;
        $settings['remind_minutes'] = max(BWC_Abandoned_Cart::MIN_REMIND, $minutes);

        // Clear and reschedule cron
        wp_clear_scheduled_hook(BWC_Abandoned_Cart::CRON_HOOK);
        wp_schedule_single_event(time() + ($settings['remind_minutes'] * MINUTE_IN_SECONDS), BWC_Abandoned_Cart::CRON_HOOK);

        add_settings_error(
            BWC_Abandoned_Cart::OPTION_NAME,
            'bwc_settings_saved',
            __('Settings saved.', 'botamation-woocommerce-tools'),
            'updated'
        );

        return $settings;
    }

    /**
     * Sanitize order notification settings
     */
    public function sanitize_order_notification_settings($input) {
        if (!is_array($input)) {
            $input = array();
        }

        $current = BWC_Order_Notification::instance()->get_settings();

        // Merge incoming keys (string values or empty to remove)
        foreach (array(BWC_Order_Notification::ACTION_CREATE, BWC_Order_Notification::ACTION_UPDATE, BWC_Order_Notification::ACTION_CANCEL) as $action) {
            if (array_key_exists($action, $input)) {
                $url = trim((string) $input[$action]);
                if ($url === '') {
                    unset($current[$action]);
                } else {
                    $current[$action] = esc_url_raw($url);
                }
            }
        }

        // Support explicit remove via POST flag
        if (!empty($_POST['bwc_on_remove_action'])) {
            $remove = sanitize_text_field(wp_unslash($_POST['bwc_on_remove_action']));
            if (in_array($remove, array(BWC_Order_Notification::ACTION_CREATE, BWC_Order_Notification::ACTION_UPDATE, BWC_Order_Notification::ACTION_CANCEL), true)) {
                unset($current[$remove]);
                add_settings_error(
                    BWC_Order_Notification::OPTION_NAME,
                    'bwc_on_removed',
                    __('Order notification webhook removed successfully.', 'botamation-woocommerce-tools'),
                    'updated'
                );
                return $current;
            }
        }

        add_settings_error(
            BWC_Order_Notification::OPTION_NAME,
            'bwc_on_saved',
            __('Order notification settings saved.', 'botamation-woocommerce-tools'),
            'updated'
        );

        return $current;
    }

    /**
     * Render tools page
     */
    public function render_tools_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $acn_url = esc_url(admin_url('admin.php?page=bwc-abandoned-cart'));
        $on_url = esc_url(admin_url('admin.php?page=bwc-order-notification'));
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Botamation Tools', 'botamation-woocommerce-tools'); ?></h1>
            <p><?php esc_html_e('Welcome to Botamation Tools. Choose a tool from the options below:', 'botamation-woocommerce-tools'); ?></p>
            
            <div class="card" style="max-width: 600px;">
                <h2><?php esc_html_e('Available Tools', 'botamation-woocommerce-tools'); ?></h2>
                <ul>
                    <li>
                        <strong><a href="<?php echo $acn_url; ?>"><?php esc_html_e('Abandoned Cart Notification', 'botamation-woocommerce-tools'); ?></a></strong><br>
                        <em><?php esc_html_e('Configure webhook notifications for abandoned shopping carts with customizable delays.', 'botamation-woocommerce-tools'); ?></em>
                    </li>
                    <li style="margin-top: 15px;">
                        <strong><a href="<?php echo $on_url; ?>"><?php esc_html_e('Order Notification', 'botamation-woocommerce-tools'); ?></a></strong><br>
                        <em><?php esc_html_e('Set up webhooks for order lifecycle events: creation, updates, and cancellations.', 'botamation-woocommerce-tools'); ?></em>
                    </li>
                </ul>
            </div>
        </div>
        <?php
    }

    /**
     * Render abandoned cart page
     */
    public function render_abandoned_cart_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Handle manual cron trigger
        if (isset($_POST['bwc_trigger_cron']) && wp_verify_nonce($_POST['_wpnonce'] ?? '', 'bwc_trigger_cron')) {
            // Run cron handler once
            BWC_Abandoned_Cart::instance()->handle_cron();
            echo '<div class="notice notice-info"><p>' . esc_html__('Cron job executed manually. Check debug log for details.', 'botamation-woocommerce-tools') . '</p></div>';
        }

        // Handle test webhook
        if (isset($_POST['bwc_test_webhook']) && wp_verify_nonce($_POST['_wpnonce'], 'bwc_test_webhook')) {
            $result = BWC_Abandoned_Cart::instance()->send_test_webhook();
            if ($result['success']) {
                echo '<div class="notice notice-success"><p>' . esc_html($result['message']) . '</p></div>';
            } else {
                echo '<div class="notice notice-error"><p>' . esc_html($result['message']) . '</p></div>';
            }
        }

        $settings = BWC_Abandoned_Cart::instance()->get_settings();
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Abandoned Cart Notification', 'botamation-woocommerce-tools'); ?></h1>
            <style>
                .bwc-inline-controls { display: flex; align-items: center; gap: 10px; flex-wrap: wrap; }
                .bwc-inline-controls label { font-weight: 600; }
                .bwc-inline-controls .regular-text { min-width: 340px; }
            </style>
            <form method="post" action="options.php" id="bwc-abandoned-cart-form">
                <?php settings_fields(BWC_Abandoned_Cart::OPTION_GROUP); ?>
                <div class="bwc-inline-controls">
                    <label for="webhook_url"><?php esc_html_e('Webhook URL', 'botamation-woocommerce-tools'); ?></label>
                    <input type="url" 
                           id="webhook_url" 
                           name="<?php echo BWC_Abandoned_Cart::OPTION_NAME; ?>[webhook_url]" 
                           value="<?php echo esc_attr($settings['webhook_url']); ?>" 
                           placeholder="https://example.com/webhook" 
                           class="regular-text" />

                    <label for="remind_minutes" style="margin-left:8px;"><?php esc_html_e('Notify After (minutes)', 'botamation-woocommerce-tools'); ?></label>
                    <input type="number" 
                           id="remind_minutes" 
                           name="<?php echo BWC_Abandoned_Cart::OPTION_NAME; ?>[remind_minutes]" 
                           value="<?php echo esc_attr($settings['remind_minutes']); ?>" 
                           min="<?php echo BWC_Abandoned_Cart::MIN_REMIND; ?>" 
                           step="1" 
                           class="small-text" />

                    <button type="submit" class="button button-primary" id="bwc-save-btn" disabled>
                        <?php esc_html_e('Save', 'botamation-woocommerce-tools'); ?>
                    </button>
                    <button type="submit" class="button" id="bwc-test-btn" form="bwc-test-form" disabled>
                        <?php esc_html_e('Send Test Data', 'botamation-woocommerce-tools'); ?>
                    </button>
                </div>
                <p class="description" style="margin-top:8px;"><?php esc_html_e('Delay before sending reminders.', 'botamation-woocommerce-tools'); ?></p>
            </form>

            <form method="post" id="bwc-test-form" style="display: none;">
                <?php wp_nonce_field('bwc_test_webhook'); ?>
                <input type="hidden" name="bwc_test_webhook" value="1" />
            </form>

            <script>
            (function() {
                var form = document.getElementById('bwc-abandoned-cart-form');
                var webhookInput = document.getElementById('webhook_url');
                var remindInput = document.getElementById('remind_minutes');
                var saveBtn = document.getElementById('bwc-save-btn');
                var testBtn = document.getElementById('bwc-test-btn');
                
                var initialWebhook = webhookInput.value;
                var initialRemind = remindInput.value;
                
                function updateButtons() {
                    var webhookChanged = webhookInput.value !== initialWebhook;
                    var remindChanged = remindInput.value !== initialRemind;
                    var hasWebhook = webhookInput.value.trim() !== '';
                    
                    saveBtn.disabled = !(webhookChanged || remindChanged);
                    testBtn.disabled = !hasWebhook || webhookChanged || remindChanged;
                }
                
                webhookInput.addEventListener('input', updateButtons);
                remindInput.addEventListener('input', updateButtons);
                document.addEventListener('DOMContentLoaded', updateButtons);
                updateButtons();
            })();
            </script>
            <hr style="margin:24px 0;" />

            <h2><?php esc_html_e('Debug Tools', 'botamation-woocommerce-tools'); ?></h2>
            <form method="post" style="margin-bottom: 20px;">
                <?php wp_nonce_field('bwc_trigger_cron'); ?>
                <input type="hidden" name="bwc_trigger_cron" value="1" />
                <input type="submit" class="button" value="<?php echo esc_attr__('Trigger Webhook Manually', 'botamation-woocommerce-tools'); ?>" />
                <p class="description"><?php esc_html_e('Runs the abandoned cart check immediately.', 'botamation-woocommerce-tools'); ?></p>
            </form>

            <?php
            // Recent cart records (last 10)
            global $wpdb;
            $table = $wpdb->prefix . BWC_Abandoned_Cart::TABLE_SLUG;
            $rows = $wpdb->get_results("SELECT * FROM {$table} ORDER BY last_updated DESC LIMIT 10", ARRAY_A);
            if (!empty($rows)) {
                echo '<h2>' . esc_html__('Recent Cart Records', 'botamation-woocommerce-tools') . '</h2>';
                echo '<table class="wp-list-table widefat fixed striped">';
                echo '<thead><tr>';
                echo '<th>' . esc_html__('ID', 'botamation-woocommerce-tools') . '</th>';
                echo '<th>' . esc_html__('Phone', 'botamation-woocommerce-tools') . '</th>';
                echo '<th>' . esc_html__('Email', 'botamation-woocommerce-tools') . '</th>';
                echo '<th>' . esc_html__('Status', 'botamation-woocommerce-tools') . '</th>';
                echo '<th>' . esc_html__('Last Updated (Site Time)', 'botamation-woocommerce-tools') . '</th>';
                echo '<th>' . esc_html__('Last Error', 'botamation-woocommerce-tools') . '</th>';
                echo '</tr></thead><tbody>';
                foreach ($rows as $r) {
                    echo '<tr>';
                    echo '<td>' . esc_html($r['id']) . '</td>';
                    echo '<td>' . esc_html($r['contact_phone'] ?: '(empty)') . '</td>';
                    echo '<td>' . esc_html($r['contact_email'] ?: '(empty)') . '</td>';
                    echo '<td>' . esc_html($r['status']) . '</td>';
                    echo '<td>' . esc_html(get_date_from_gmt($r['last_updated'], 'Y-m-d H:i:s')) . '</td>';
                    echo '<td>' . esc_html($r['last_error'] ?: '') . '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';
            }
            ?>
        </div>
        <?php
    }

    /**
     * Render order notification page
     */
    public function render_order_notification_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Handle removal request
        if (isset($_POST['bwc_on_remove_action'])) {
            $action = sanitize_text_field(wp_unslash($_POST['bwc_on_remove_action']));
            if (in_array($action, array(BWC_Order_Notification::ACTION_CREATE, BWC_Order_Notification::ACTION_UPDATE, BWC_Order_Notification::ACTION_CANCEL), true)) {
                if (wp_verify_nonce($_POST['_wpnonce'], 'bwc_on_remove_' . $action)) {
                    $settings = BWC_Order_Notification::instance()->get_settings();
                    unset($settings[$action]);
                    update_option(BWC_Order_Notification::OPTION_NAME, $settings);
                    echo '<div class="notice notice-success"><p>' . esc_html__('Order notification webhook removed successfully.', 'botamation-woocommerce-tools') . '</p></div>';
                } else {
                    echo '<div class="notice notice-error"><p>' . esc_html__('Security check failed. Please try again.', 'botamation-woocommerce-tools') . '</p></div>';
                }
            }
        }

        // Handle test webhook submission per action
        if (isset($_POST['bwc_on_test_action']) && check_admin_referer('bwc_on_test_webhook')) {
            $action = sanitize_text_field(wp_unslash($_POST['bwc_on_test_action']));
            $result = BWC_Order_Notification::instance()->send_test_webhook($action);
            if ($result['success']) {
                echo '<div class="notice notice-success"><p>' . esc_html($result['message']) . '</p></div>';
            } else {
                echo '<div class="notice notice-error"><p>' . esc_html($result['message']) . '</p></div>';
            }
        }

        $settings = BWC_Order_Notification::instance()->get_settings();
        $hasCreate = array_key_exists(BWC_Order_Notification::ACTION_CREATE, $settings);
        $hasUpdate = array_key_exists(BWC_Order_Notification::ACTION_UPDATE, $settings);
        $hasCancel = array_key_exists(BWC_Order_Notification::ACTION_CANCEL, $settings);
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Order Notification', 'botamation-woocommerce-tools'); ?></h1>
            <style>
                .on-inline { display:flex; align-items:center; gap:10px; flex-wrap:wrap; }
                .on-section { border:1px solid #ccd0d4; padding:12px; margin-top:12px; background:#fff; }
                .on-row { display:flex; align-items:center; gap:10px; }
                .on-row .regular-text { min-width:340px; }
                .on-remove { color:#b32d2e !important; text-decoration:none; display:inline-flex; align-items:center; padding:0; border:none; background:none; cursor:pointer; }
                .on-remove:hover { color:#86181a !important; }
                .on-remove .dashicons { font-size:16px; }
                .on-remove-form { display:inline; margin:0; }
                .on-main-form { display:flex; align-items:center; gap:10px; }
                .hidden { display:none; }
            </style>

            <div class="on-inline">
                <label for="on-action-select" style="font-weight:600;"><?php esc_html_e('Choose Action', 'botamation-woocommerce-tools'); ?></label>
                <select id="on-action-select" class="regular-text">
                    <option value="">-- <?php esc_html_e('Select', 'botamation-woocommerce-tools'); ?> --</option>
                    <option value="<?php echo esc_attr(BWC_Order_Notification::ACTION_CREATE); ?>" <?php selected($hasCreate); ?> <?php disabled($hasCreate); ?>><?php esc_html_e('Order Create', 'botamation-woocommerce-tools'); ?></option>
                    <option value="<?php echo esc_attr(BWC_Order_Notification::ACTION_UPDATE); ?>" <?php selected($hasUpdate); ?> <?php disabled($hasUpdate); ?>><?php esc_html_e('Order Update', 'botamation-woocommerce-tools'); ?></option>
                    <option value="<?php echo esc_attr(BWC_Order_Notification::ACTION_CANCEL); ?>" <?php selected($hasCancel); ?> <?php disabled($hasCancel); ?>><?php esc_html_e('Order Cancel', 'botamation-woocommerce-tools'); ?></option>
                </select>
                <button class="button" id="on-add-action"><?php esc_html_e('Add', 'botamation-woocommerce-tools'); ?></button>
            </div>

            <div id="on-sections">
                <?php $this->render_action_section(BWC_Order_Notification::ACTION_CREATE, $settings); ?>
                <?php $this->render_action_section(BWC_Order_Notification::ACTION_UPDATE, $settings); ?>
                <?php $this->render_action_section(BWC_Order_Notification::ACTION_CANCEL, $settings); ?>
            </div>

            <template id="on-section-template">
                <?php $this->render_action_section('__ACTION__', array()); ?>
            </template>

            <script>
            (function(){
                var select = document.getElementById('on-action-select');
                var addBtn = document.getElementById('on-add-action');
                var sections = document.getElementById('on-sections');

                function sectionId(action){ return 'on-section-' + action; }
                function getSection(action){ return document.getElementById(sectionId(action)); }

                function initSection(sec){
                    var input = sec.querySelector('input[type="url"]');
                    if (!input) return;
                    var initial = input.getAttribute('data-initial') || input.value || '';
                    var saveBtn = sec.querySelector('[data-on-save]');
                    var testBtn = sec.querySelector('[data-on-test]');
                    var removeBtn = sec.querySelector('.on-remove');
                    
                    function update(){
                        var current = (input.value || '').trim();
                        var changed = current !== (initial || '').trim();
                        if (saveBtn) saveBtn.disabled = !changed;
                        if (testBtn) testBtn.disabled = changed || current === '';
                    }
                    input.addEventListener('input', update);
                    update();
                    
                    // Handle remove button click
                    if (removeBtn) {
                        removeBtn.addEventListener('click', function(e) {
                            e.preventDefault();
                            var action = removeBtn.closest('.on-remove-form').querySelector('input[name="bwc_on_remove_action"]').value;
                            var actionLabel = removeBtn.closest('.on-section').querySelector('strong').textContent.trim();
                            
                            // Ask for confirmation
                            if (!confirm('Are you sure you want to remove the ' + actionLabel + ' webhook? This will permanently delete the saved configuration.')) {
                                return;
                            }
                            
                            // Submit the remove form
                            var removeForm = removeBtn.closest('.on-remove-form');
                            if (removeForm) {
                                removeForm.submit();
                            }
                        });
                    }
                }

                // Initialize existing sections
                Array.prototype.forEach.call(document.querySelectorAll('.on-section'), initSection);

                addBtn.addEventListener('click', function(e){
                    e.preventDefault();
                    var action = (select.value || '').trim();
                    if (!action) return;
                    var existing = getSection(action);
                    if (existing) {
                        // If the section exists (likely hidden by default), unhide and init it
                        existing.classList.remove('hidden');
                        initSection(existing);
                        // Disable the selected option to prevent duplicate adds
                        var opt = select.querySelector('option[value="' + action + '"]');
                        if (opt) opt.disabled = true;
                        return;
                    }
                    var tpl = document.getElementById('on-section-template');
                    var html = tpl.innerHTML.replace(/__ACTION__/g, action);
                    // Create wrapper element to parse template HTML
                    var div = document.createElement('div');
                    div.innerHTML = html;
                    var newSection = div.firstElementChild;
                    // Unhide newly created section
                    newSection.classList.remove('hidden');
                    sections.appendChild(newSection);
                    initSection(newSection);
                    // Disable the selected option to prevent duplicate adds
                    var opt2 = select.querySelector('option[value="' + action + '"]');
                    if (opt2) opt2.disabled = true;
                });
            })();
            </script>
        </div>
        <?php
    }

    /**
     * Render action section
     */
    private function render_action_section($action, $settings) {
        $map = array(
            BWC_Order_Notification::ACTION_CREATE => __('Order Create', 'botamation-woocommerce-tools'),
            BWC_Order_Notification::ACTION_UPDATE => __('Order Update', 'botamation-woocommerce-tools'),
            BWC_Order_Notification::ACTION_CANCEL => __('Order Cancel', 'botamation-woocommerce-tools'),
        );

        $key = (string) $action;
        $exists = isset($settings[$key]) && $settings[$key];
        $label = isset($map[$key]) ? $map[$key] : strtoupper($key);
        $url = $exists ? $settings[$key] : '';
        $hiddenClass = in_array($key, array(BWC_Order_Notification::ACTION_CREATE, BWC_Order_Notification::ACTION_UPDATE, BWC_Order_Notification::ACTION_CANCEL), true) ? ($exists ? '' : ' hidden') : ' hidden';
        ?>
        <div class="on-section<?php echo esc_attr($hiddenClass); ?>" id="on-section-<?php echo esc_attr($key); ?>">
            <div class="on-row">
                <form method="post" action="options.php" class="on-main-form">
                    <?php settings_fields(BWC_Order_Notification::OPTION_GROUP); ?>
                    <strong style="min-width:140px; display:inline-block;">&nbsp;<?php echo esc_html($label); ?></strong>
                    <input type="url" class="regular-text" data-initial="<?php echo esc_attr($url); ?>" name="<?php echo esc_attr(BWC_Order_Notification::OPTION_NAME); ?>[<?php echo esc_attr($key); ?>]" value="<?php echo esc_attr($url); ?>" placeholder="https://example.com/webhook" required />
                    <button type="submit" class="button button-primary" data-on-save disabled><?php esc_html_e('Save', 'botamation-woocommerce-tools'); ?></button>
                    <button type="submit" class="button" data-on-test form="on-test-form-<?php echo esc_attr($key); ?>" disabled><?php esc_html_e('Send Test Data', 'botamation-woocommerce-tools'); ?></button>
                </form>
                <form method="post" class="on-remove-form">
                    <?php wp_nonce_field('bwc_on_remove_' . $key); ?>
                    <input type="hidden" name="bwc_on_remove_action" value="<?php echo esc_attr($key); ?>" />
                    <button type="submit" class="button-link on-remove" title="<?php esc_attr_e('Remove', 'botamation-woocommerce-tools'); ?>">
                        <span class="dashicons dashicons-no"></span>
                    </button>
                </form>
            </div>
            <form method="post" id="on-test-form-<?php echo esc_attr($key); ?>" class="hidden">
                <?php wp_nonce_field('bwc_on_test_webhook'); ?>
                <input type="hidden" name="bwc_on_test_action" value="<?php echo esc_attr($key); ?>" />
            </form>
        </div>
        <?php
    }
}