<?php
/**
 * Order Notification functionality
 *
 * @package Botamation_WooCommerce_Tools
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * BWC_Order_Notification class
 */
class BWC_Order_Notification {

    /**
     * Instance
     *
     * @var BWC_Order_Notification
     */
    private static $instance = null;

    /**
     * Constants
     */
    const OPTION_NAME = 'bwc_order_notification_settings';
    const OPTION_GROUP = 'bwc_order_notification_group';

    // Post meta flags to prevent duplicate sends
    const META_SENT_CREATE = '_bwc_on_sent_create';
    const META_SENT_CANCEL = '_bwc_on_sent_cancel';
    const META_LAST_STATUS = '_bwc_on_last_status';

    // Actions keys
    const ACTION_CREATE = 'create';
    const ACTION_UPDATE = 'update';
    const ACTION_CANCEL = 'cancel';

    /**
     * Get instance
     *
     * @return BWC_Order_Notification
     */
    public static function instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // WooCommerce hooks
        add_action('woocommerce_new_order', array($this, 'handle_order_created'), 20, 1);
        add_action('woocommerce_order_status_changed', array($this, 'handle_order_status_changed'), 20, 4);
    }

    /**
     * Handle order created
     */
    public function handle_order_created($order_id) {
        $settings = $this->get_settings();
        if (empty($settings[self::ACTION_CREATE])) {
            return;
        }

        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }

        if ($order->get_meta(self::META_SENT_CREATE)) {
            return;
        }

        $this->send_order_webhook(self::ACTION_CREATE, $order, $settings[self::ACTION_CREATE]);
        $order->update_meta_data(self::META_SENT_CREATE, current_time('mysql', true));
        $order->save();
    }

    /**
     * Handle order status changed
     */
    public function handle_order_status_changed($order_id, $old_status, $new_status, $order) {
        $settings = $this->get_settings();
        if (!$order instanceof WC_Order) {
            $order = wc_get_order($order_id);
        }
        if (!$order) {
            return;
        }

        // Cancelled
        if ($new_status === 'cancelled' && !empty($settings[self::ACTION_CANCEL])) {
            if (!$order->get_meta(self::META_SENT_CANCEL)) {
                $this->send_order_webhook(self::ACTION_CANCEL, $order, $settings[self::ACTION_CANCEL]);
                $order->update_meta_data(self::META_SENT_CANCEL, current_time('mysql', true));
                $order->save();
            }
            // No further update send for this transition
            return;
        }

        // Updates on any other status change
        if (!empty($settings[self::ACTION_UPDATE])) {
            $last = (string) $order->get_meta(self::META_LAST_STATUS);
            if ($last !== $new_status) {
                $this->send_order_webhook(self::ACTION_UPDATE, $order, $settings[self::ACTION_UPDATE]);
                $order->update_meta_data(self::META_LAST_STATUS, $new_status);
                $order->save();
            }
        }
    }

    /**
     * Send order webhook
     */
    private function send_order_webhook($action, $order, $webhook) {
        $payload = $this->build_order_payload($order, $action);
        $body = wp_json_encode($payload);
        if (false === $body) {
            return;
        }

        $response = wp_remote_post($webhook, array(
            'timeout' => 15,
            'headers' => array('Content-Type' => 'application/json'),
            'body' => $body,
        ));

        if (is_wp_error($response)) {
            error_log('BWC Order Notify: Webhook error (' . $action . '): ' . $response->get_error_message());
            return;
        }

        $code = wp_remote_retrieve_response_code($response);
        error_log('BWC Order Notify: Webhook (' . $action . ') response code: ' . $code);
    }

    /**
     * Build order payload
     */
    private function build_order_payload($order, $action) {
        $billing = array(
            'first_name' => $order->get_billing_first_name(),
            'last_name' => $order->get_billing_last_name(),
            'company' => $order->get_billing_company(),
            'email' => $order->get_billing_email(),
            'phone' => $order->get_billing_phone(),
            'address_1' => $order->get_billing_address_1(),
            'address_2' => $order->get_billing_address_2(),
            'city' => $order->get_billing_city(),
            'state' => $order->get_billing_state(),
            'postcode' => $order->get_billing_postcode(),
            'country' => $order->get_billing_country(),
        );

        $shipping = array(
            'first_name' => $order->get_shipping_first_name(),
            'last_name' => $order->get_shipping_last_name(),
            'company' => $order->get_shipping_company(),
            'email' => $order->get_billing_email(),
            'phone' => $order->get_billing_phone(),
            'address_1' => $order->get_shipping_address_1(),
            'address_2' => $order->get_shipping_address_2(),
            'city' => $order->get_shipping_city(),
            'state' => $order->get_shipping_state(),
            'postcode' => $order->get_shipping_postcode(),
            'country' => $order->get_shipping_country(),
        );

        $items_payload = array();
        $products_total_with_tax = 0.0;

        foreach ($order->get_items('line_item') as $item) {
            if (!$item instanceof WC_Order_Item_Product) {
                continue;
            }

            $product = $item->get_product();
            $product_id = $product ? $product->get_id() : 0;
            $qty = (int) $item->get_quantity();
            $line_total = (float) $item->get_total();
            $line_tax = (float) $item->get_total_tax();
            $products_total_with_tax += ($line_total + $line_tax);

            $price_each = $qty > 0 ? ($line_total / $qty) : 0.0;
            $image = '';
            if ($product && $product->get_image_id()) {
                $image = wp_get_attachment_image_url($product->get_image_id(), 'full');
            }
            if (!$image) {
                $image = wc_placeholder_img_src();
            }

            $attrs = array();
            if ($product && ($product instanceof WC_Product_Variation)) {
                $attrs = $product->get_attributes();
            }

            $items_payload[] = array(
                'product_id' => (int) $product_id,
                'variation_id' => ($product && $product instanceof WC_Product_Variation) ? (int) $product->get_id() : 0,
                'name' => $item->get_name(),
                'price' => (float) wc_format_decimal($price_each, 2),
                'quantity' => $qty,
                'subtotal' => (float) wc_format_decimal($line_total, 2),
                'image' => $image,
                'variation' => $attrs,
            );
        }

        // Invoice URL best-effort
        $invoice_url = method_exists($order, 'get_view_order_url') ? $order->get_view_order_url() : $order->get_checkout_order_received_url();

        // Build contact object - prefer billing, fallback to shipping for each field
        $contact = array(
            'phone' => $order->get_billing_phone(),
            'email' => $order->get_billing_email(),
            'first_name' => $order->get_billing_first_name(),
            'last_name' => $order->get_billing_last_name(),
        );

        // Fallback to shipping if billing is empty
        if (empty($contact['phone'])) {
            $contact['phone'] = $billing['phone'] ?? ($shipping['phone'] ?? '');
        }
        if (empty($contact['email'])) {
            $contact['email'] = $billing['email'] ?? ($shipping['email'] ?? '');
        }
        if (empty($contact['first_name'])) {
            $contact['first_name'] = $order->get_shipping_first_name();
        }
        if (empty($contact['last_name'])) {
            $contact['last_name'] = $order->get_shipping_last_name();
        }

        return array(
            'generated_at' => current_time('mysql', true),
            'action' => $action,
            'order_id' => $order->get_id(),
            'order_number' => $order->get_order_number(),
            'status' => $order->get_status(),
            'currency' => $order->get_currency(),
            'billing' => $billing,
            'shipping' => $shipping,
            'contact' => $contact,
            'items' => $items_payload,
            'total_products_with_tax' => (float) wc_format_decimal($products_total_with_tax, 2),
            'invoice_url' => $invoice_url,
        );
    }

    /**
     * Get settings
     */
    public function get_settings() {
        $settings = get_option(self::OPTION_NAME, array());
        return is_array($settings) ? $settings : array();
    }

    /**
     * Send test webhook
     */
    public function send_test_webhook($action) {
        $valid = array(self::ACTION_CREATE, self::ACTION_UPDATE, self::ACTION_CANCEL);
        if (!in_array($action, $valid, true)) {
            return array(
                'success' => false,
                'message' => __('Invalid action for test.', 'botamation-woocommerce-tools')
            );
        }

        $settings = $this->get_settings();
        if (empty($settings[$action])) {
            return array(
                'success' => false,
                'message' => __('Please configure a webhook URL for this action first.', 'botamation-woocommerce-tools')
            );
        }

        // Build a synthetic test payload similar to real one
        $test_payload = array(
            'test' => true,
            'generated_at' => current_time('mysql', true),
            'action' => $action,
            'order_id' => 'test_' . wp_generate_uuid4(),
            'order_number' => 'T-' . strtoupper(wp_generate_uuid4()),
            'status' => $action === self::ACTION_CANCEL ? 'cancelled' : 'processing',
            'currency' => get_woocommerce_currency(),
            'billing' => array(
                'first_name' => 'John',
                'last_name' => 'Doe',
                'company' => 'Test Co',
                'email' => 'john.doe@example.com',
                'phone' => '+1234567890',
                'address_1' => '123 Test St',
                'address_2' => '',
                'city' => 'Test City',
                'state' => 'TS',
                'postcode' => '12345',
                'country' => 'US',
            ),
            'shipping' => array(
                'first_name' => 'Jane',
                'last_name' => 'Smith',
                'company' => 'Shipping Co',
                'email' => 'jane.smith@example.com',
                'phone' => '+0987654321',
                'address_1' => '456 Ship Rd',
                'address_2' => '',
                'city' => 'Ship City',
                'state' => 'SC',
                'postcode' => '67890',
                'country' => 'US',
            ),
            'contact' => array(
                'phone' => '+1234567890',
                'email' => 'john.doe@example.com',
                'first_name' => 'John',
                'last_name' => 'Doe',
            ),
            'items' => array(
                array(
                    'product_id' => 1,
                    'variation_id' => 0,
                    'name' => 'Sample Product',
                    'price' => 19.99,
                    'quantity' => 2,
                    'subtotal' => 39.98,
                    'image' => 'https://via.placeholder.com/300x300.jpg',
                    'variation' => array(),
                ),
                array(
                    'product_id' => 2,
                    'variation_id' => 0,
                    'name' => 'Another Item',
                    'price' => 9.99,
                    'quantity' => 1,
                    'subtotal' => 9.99,
                    'image' => 'https://via.placeholder.com/300x300.jpg',
                    'variation' => array(),
                ),
            ),
        );

        // Approximate products total with tax (+10%)
        $sum = 0.0;
        foreach ($test_payload['items'] as $it) {
            $sum += (float) $it['subtotal'];
        }
        $test_payload['total_products_with_tax'] = round($sum * 1.10, 2);
        $test_payload['invoice_url'] = home_url('/my-account/view-order/0?test=1');

        $body = wp_json_encode($test_payload);
        if (false === $body) {
            return array(
                'success' => false,
                'message' => __('Failed to encode test payload.', 'botamation-woocommerce-tools')
            );
        }

        $response = wp_remote_post($settings[$action], array(
            'timeout' => 15,
            'headers' => array(
                'Content-Type' => 'application/json',
                'User-Agent' => 'BWC-Order-Test-Webhook/1.0'
            ),
            'body' => $body,
        ));

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => sprintf(__('Webhook test failed: %s', 'botamation-woocommerce-tools'), $response->get_error_message())
            );
        }

        $code = wp_remote_retrieve_response_code($response);
        $respBody = wp_remote_retrieve_body($response);
        if ($code < 200 || $code >= 300) {
            return array(
                'success' => false,
                'message' => sprintf(__('Webhook test failed with HTTP %d: %s', 'botamation-woocommerce-tools'), $code, $respBody)
            );
        }

        return array(
            'success' => true,
            'message' => sprintf(__('Test webhook sent successfully! Response: HTTP %d', 'botamation-woocommerce-tools'), $code)
        );
    }
}